// src/contentScript.js

// YouTube 동영상 ID를 추출하는 함수
function extractYoutubeId(url) {
  const regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|\&v=)([^#\&\?]*).*/;
  const match = url.match(regExp);
  return (match && match[2].length === 11) ? match[2] : null;
}

// 알림 메시지 표시 함수
function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.className = 'teetube-notification';
  notification.textContent = message;
  document.body.appendChild(notification);

  // 3초 후 알림 제거
  setTimeout(() => {
    notification.style.opacity = '0';
    setTimeout(() => notification.remove(), 300);
  }, 3000);
}

// TeeTube 버튼 생성 함수
function createTeeTubeButton(videoId) {
  const button = document.createElement('button');
  button.className = 'teetube-btn';
  button.innerHTML = `
    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
      <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
    </svg>
    TeeTube
  `;

  button.addEventListener('click', async (e) => {
    e.preventDefault();
    e.stopPropagation();

    // 버튼 비활성화 (중복 클릭 방지)
    button.disabled = true;
    button.style.opacity = '0.5';

    try {
      // background script에 메시지 전송
      chrome.runtime.sendMessage({
        action: 'ADD_TO_TEETUBE',
        videoId: videoId
      }, response => {
        if (chrome.runtime.lastError) {
          console.log('Message sending failed:', chrome.runtime.lastError.message);
          // 에러가 발생해도 사용자 경험에 영향을 주지 않도록 조용히 처리
        }

        // 버튼 다시 활성화
        button.disabled = false;
        button.style.opacity = '1';
      });
    } catch (error) {
      console.error('Error:', error);
      // 버튼 다시 활성화
      button.disabled = false;
      button.style.opacity = '1';
    }
  });

  return button;
}

// 임베드된 YouTube 동영상을 찾고 버튼 추가
function findAndAddButtons() {
  // iframe 엘리먼트 찾기
  const iframes = document.querySelectorAll('iframe[src*="youtube.com/embed"]');

  iframes.forEach(iframe => {
    // 이미 처리된 iframe은 건너뛰기
    if (iframe.parentElement.classList.contains('teetube-container')) {
      return;
    }

    const videoId = extractYoutubeId(iframe.src);
    if (!videoId) return;

    // 원래 iframe의 스타일과 속성을 보존
    const originalWidth = iframe.width || iframe.style.width;
    const originalHeight = iframe.height || iframe.style.height;
    const originalStyle = window.getComputedStyle(iframe);

    // iframe을 컨테이너로 감싸기
    const container = document.createElement('div');
    container.className = 'teetube-container';

    // 원래 크기를 컨테이너에 적용
    if (originalWidth) container.style.width = originalWidth.includes('px') ? originalWidth : originalWidth + 'px';
    if (originalHeight) container.style.height = originalHeight.includes('px') ? originalHeight : originalHeight + 'px';

    // 원래 iframe의 aspect ratio 유지
    if (originalStyle.aspectRatio) {
      container.style.aspectRatio = originalStyle.aspectRatio;
    }

    iframe.parentElement.insertBefore(container, iframe);
    container.appendChild(iframe);

    // TeeTube 버튼 추가
    const button = createTeeTubeButton(videoId);
    container.appendChild(button);
  });
}

// 페이지 로드 시 실행
findAndAddButtons();

// DOM 변경 감지를 위한 MutationObserver 설정
const observer = new MutationObserver((mutations) => {
  mutations.forEach((mutation) => {
    if (mutation.addedNodes.length) {
      findAndAddButtons();
    }
  });
});

// 전체 문서에 대해 DOM 변경 감지 시작
observer.observe(document.body, {
  childList: true,
  subtree: true
});