// src/popup.js

import { auth, db } from "./firebase.js";
import {
  signInWithEmailAndPassword,
  onAuthStateChanged,
  signOut,
  createUserWithEmailAndPassword,
  sendPasswordResetEmail
} from "firebase/auth";
import {
  collection,
  query,
  where,
  getDocs,
  doc,
  setDoc,
  orderBy,
  deleteDoc
} from "firebase/firestore";

/*
 * 외부 라이브러리(예: ytdl-core, youtube-metadata 등)를 통해
 * 현재 URL에 대한 실제 메타데이터(artist, title, duration...)를 가져온다고 가정.
 * 여기서는 일단 placeholder 함수를 예시로 두겠습니다.
 */
async function fetchYoutubeMetadata(videoId) {
  const apiKey = "AIzaSyChcbo-VasFOwQkZpysPr9qZc9Nv-331gU"; // 실제 발급받은 API 키
  const url = `https://www.googleapis.com/youtube/v3/videos?id=${videoId}&part=snippet,contentDetails&key=${apiKey}`;

  const response = await fetch(url);
  const data = await response.json();

  if (!data.items || data.items.length === 0) {
    throw new Error("No video found for the given ID.");
  }

  const snippet = data.items[0].snippet;
  const contentDetails = data.items[0].contentDetails;

  // 채널명 = snippet.channelTitle, 제목 = snippet.title
  const title = snippet.title;
  const artist = snippet.channelTitle;

  // PT4M15S 같은 ISO 8601 duration → 0:04:15 형식으로 변환해주기
  const duration = parseIso8601Duration(contentDetails.duration);

  // 썸네일은 snippet.thumbnails.* 에서 원하는 해상도 선택
  const thumbnailUrl = snippet.thumbnails.high?.url || snippet.thumbnails.default?.url;

  return {
    title,
    artist,
    duration,
    thumbnailUrl
  };
}

/**
 * ISO 8601 형식(예: "PT4M15S")을 "0:04:15" 형태로 변환
 */
function parseIso8601Duration(isoDuration) {
  // 정규식으로 시/분/초 추출
  const match = isoDuration.match(/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/);
  if (!match) return isoDuration;

  const hours = parseInt(match[1] || "0", 10);
  const minutes = parseInt(match[2] || "0", 10);
  const seconds = parseInt(match[3] || "0", 10);

  // 예: 시(hour)가 없으면 "분:초"만
  if (hours > 0) {
    return `${hours}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
  } else {
    return `${minutes}:${String(seconds).padStart(2, "0")}`;
  }
}


// 로그인 버튼 클릭
document.getElementById("login-btn").addEventListener("click", async () => {
  const email = document.getElementById("email").value;
  const password = document.getElementById("password").value;
  try {
    await signInWithEmailAndPassword(auth, email, password);
    // 로그인 성공 시 onAuthStateChanged에서 화면 전환
  } catch (error) {
    document.getElementById("login-status").innerText =
      `Login failed: ${error.message}`;
  }
});

// Firebase 인증 상태 변화 감지
onAuthStateChanged(auth, async (user) => {
  console.log('[Auth] 인증 상태 변경:', user ? '로그인됨' : '로그인되지 않음');

  document.getElementById("loading-section").style.display = "none";
  document.getElementById("login-section").style.display = "none";
  document.getElementById("signup-section").style.display = "none";

  if (user) {
    document.getElementById("login-section").style.display = "none";
    document.getElementById("app-section").style.display = "flex";
    document.querySelector(".fab-container").style.display = "block";

    const email = user.email || "";
    const firstChar = email.charAt(0).toUpperCase() || "?";
    const userAvatar = document.getElementById("user-avatar");
    const userMenu = document.getElementById("user-menu");
    const userEmailText = document.getElementById("user-email");

    userAvatar.innerText = firstChar;
    userAvatar.title = email;
    userEmailText.innerText = email;

    // 아바타 클릭 -> 메뉴 토글
    userAvatar.onclick = () => {
      if (userMenu.style.display === "none") {
        userMenu.style.display = "block";
      } else {
        userMenu.style.display = "none";
      }
    };

    await loadDownloadQueue(user.uid); // 목록 로딩
  } else {
    document.getElementById("login-section").style.display = "block";
    document.getElementById("app-section").style.display = "none";
    document.querySelector(".fab-container").style.display = "none";
  }
});

// 로그아웃 버튼
document.getElementById("logout-btn").addEventListener("click", async () => {
  try {
    await signOut(auth);
    console.log("로그아웃 완료");
  } catch (err) {
    console.error("로그아웃 실패:", err);
  }
});


// "현재 탭의 YouTube URL 저장" 버튼 클릭
document.getElementById("save-btn").addEventListener("click", () => {
  chrome.tabs.query({ active: true, currentWindow: true }, async (tabs) => {
    const currentTab = tabs[0];
    if (!currentTab) return;

    const url = currentTab.url;

    // 유효성 검사 - YouTube URL 패턴 수정
    if (!isValidYoutubeUrl(url)) {
      document.getElementById("message").innerText =
        "유튜브 영상 주소가 아닙니다.";
      return;
    }

    // 유튜브 ID 추출
    const videoId = extractYoutubeId(url);
    console.log("[SAVE] 추출된 videoId:", videoId);

    // 현재 로그인 유저 확인
    const user = auth.currentUser;
    if (!user) {
      console.log("[SAVE] user가 null → 로그인 필요");
      document.getElementById("message").innerText = "로그인이 필요합니다.";
      return;
    }
    console.log("[SAVE] 현재 로그인된 유저 UID:", user.uid);

    // 이미 저장된 영상인지 체크
    const exists = await isVideoAlreadySaved(user.uid, videoId);
    console.log("[SAVE] 중복 여부:", exists);
    if (exists) {
      document.getElementById("message").innerText = "이미 저장된 동영상입니다.";
      return;
    }

    // 실제 메타데이터 가져오기 (예시: fetchYoutubeMetadata)
    const meta = await fetchYoutubeMetadata(videoId);

    // Firestore 문서 ID = (최대값 - 현재시간)
    const MAX_TIMESTAMP = 9999999999999;
    const docId = (MAX_TIMESTAMP - Date.now()).toString();

    // Firestore에 저장할 데이터
    const videoData = {
      artist: meta.artist,
      duration: meta.duration,
      id: videoId,
      savedAt: new Date().toLocaleString("ko-KR", { timeZone: "Asia/Seoul" }),
      thumbnailUrl: meta.thumbnailUrl,
      timestamp: Date.now(),
      title: meta.title,
      userId: user.uid,
      youtubeUrl: url,
      docId: docId
    };

    try {
      console.log("[SAVE] saveVideoData 호출 직전:", user.uid, docId, videoData);
      await saveVideoData(user.uid, docId, videoData);
      document.getElementById("message").innerText = "동영상이 저장되었습니다.";
      console.log("[SAVE] 저장 후 loadDownloadQueue 호출");
      loadDownloadQueue(user.uid);
    } catch (error) {
      console.log("[SAVE] setDoc 에러:", error);
      document.getElementById("message").innerText = `저장 실패: ${error.message}`;
    }
  });
});

// Firestore에 데이터 저장
async function saveVideoData(uid, docId, videoData) {
  console.log("[DB] saveVideoData() 시작:", uid, docId);
  const videoDocRef = doc(db, "users", uid, "downloadQueue", docId);
  try {
    await setDoc(videoDocRef, videoData);
    console.log("[DB] Firestore setDoc 성공:", videoDocRef.path);
  } catch (error) {
    console.error("[DB] Firestore setDoc 실패:", error);
    throw error;
  }
}

// Firestore에서 목록 로딩
async function loadDownloadQueue(uid) {
  console.log("[DB] loadDownloadQueue() 시작:", uid);

  // 예: 전체 문서를 전부 가져와 총 개수만 구한다 (간단 구현).
  // 실제로는 count() 기능(집계 쿼리) 혹은 별도의 필드/컬렉션 유지 가능.
  const allSnap = await getDocs(
    query(collection(db, "users", uid, "downloadQueue"))
  );
  const totalCount = allSnap.size;

  // 메인 쿼리 (정렬 + 제한 등)
  const q = query(
    collection(db, "users", uid, "downloadQueue"),
    orderBy("savedAt", "desc")
    // limit(10) 등으로 페이지네이션 가능
  );
  const qsnap = await getDocs(q);

  document.getElementById("list-count").innerText =
    totalCount > 0 ? `(총 ${totalCount}개)` : "";

  const queueList = document.getElementById("download-queue");
  queueList.innerHTML = "";

  qsnap.forEach((snap) => {
    const data = snap.data();
    console.log("[DB] 문서:", snap.id, data);

    const li = document.createElement("li");
    li.innerHTML = `
      <div class="video-item">
        <img class="video-thumbnail" src="${data.thumbnailUrl}" alt="비디오 썸네일">
        <div class="video-info">
          <div class="video-title">${data.title}</div>
          <div class="video-meta">
            <span class="video-artist">${data.artist}</span>
            <span class="video-duration">${data.duration}</span>
          </div>
          <div class="video-actions">
            <div class="video-link">
              <a href="${data.youtubeUrl}" target="_blank">유튜브에서 보기</a>
            </div>
            <button class="delete-btn" data-doc-id="${data.docId}" title="삭제">
              <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
              </svg>
            </button>
          </div>
        </div>
      </div>
    `;

    // 삭제 버튼에 이벤트 리스너 추가
    const deleteBtn = li.querySelector(".delete-btn");
    deleteBtn.addEventListener("click", async () => {
      // 확인 대화상자 표시
      if (confirm("이 동영상을 삭제하시겠습니까?")) {
        try {
          await deleteVideo(uid, data.docId);
          await loadDownloadQueue(uid); // 목록 새로고침
          document.getElementById("message").innerText = "동영상이 삭제되었습니다.";
        } catch (error) {
          console.error("삭제 실패:", error);
          document.getElementById("message").innerText = `삭제 실패: ${error.message}`;
        }
      }
    });

    queueList.appendChild(li);
  });
}

// 비디오 삭제 함수
async function deleteVideo(uid, docId) {
  const videoDocRef = doc(db, "users", uid, "downloadQueue", docId);
  await deleteDoc(videoDocRef);
}

/**
 * 예: "0:04:15.000000" → "4:15" 형태로 변환
 */
function formatDuration(durationStr) {
  try {
    const [h, m, sWithMs] = durationStr.split(":");
    const s = sWithMs.split(".")[0]; // "15.000000" → "15"
    const hours = parseInt(h, 10);
    const minutes = parseInt(m, 10);
    const seconds = parseInt(s, 10);

    if (hours > 0) {
      return `${hours}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}`;
    } else {
      return `${minutes}:${String(seconds).padStart(2, "0")}`;
    }
  } catch (e) {
    return durationStr; // 포맷이 다를 경우 그냥 원본
  }
}



// 유튜브 URL 검사 함수 수정
function isValidYoutubeUrl(url) {
  // youtu.be 단축 URL도 포함하도록 패턴 확장
  const patterns = [
    /^(https?:\/\/)?(www\.)?(m\.)?(youtube\.com\/watch\?v=)[\w-]+/,
    /^(https?:\/\/)?(www\.)?youtu\.be\/[\w-]+/
  ];
  return patterns.some(pattern => pattern.test(url));
}

// 유튜브 ID 추출
function extractYoutubeId(url) {
  try {
    const urlObj = new URL(url);
    return urlObj.searchParams.get("v");
  } catch (error) {
    return null;
  }
}

// 중복 확인 (이미 해당 videoId로 저장된 문서가 있는지)
async function isVideoAlreadySaved(uid, videoId) {
  const q = query(
    collection(db, "users", uid, "downloadQueue"),
    where("id", "==", videoId)
  );
  const qsnap = await getDocs(q);
  return !qsnap.empty;
}

// 로그인/회원가입 섹션 전환 - 이벤트 리스너 위치 수정
document.addEventListener('DOMContentLoaded', () => {
  const showSignupLink = document.getElementById("show-signup");
  const showLoginLink = document.getElementById("show-login");

  if (showSignupLink) {
    showSignupLink.addEventListener("click", (e) => {
      e.preventDefault();
      console.log("회원가입 화면으로 전환");
      document.getElementById("login-section").style.display = "none";
      document.getElementById("signup-section").style.display = "block";
    });
  }

  if (showLoginLink) {
    showLoginLink.addEventListener("click", (e) => {
      e.preventDefault();
      console.log("로그인 화면으로 전환");
      document.getElementById("signup-section").style.display = "none";
      document.getElementById("login-section").style.display = "block";
    });
  }
});

// 회원가입 처리
document.getElementById("signup-btn").addEventListener("click", async () => {
  console.log("회원가입 버튼 클릭됨");

  const email = document.getElementById("signup-email").value;
  const password = document.getElementById("signup-password").value;
  const passwordConfirm = document.getElementById("signup-password-confirm").value;
  const statusElement = document.getElementById("signup-status");

  console.log("입력값 확인:", { email, password, passwordConfirm });

  // 입력값 검증
  if (!email || !password || !passwordConfirm) {
    statusElement.innerText = "모든 필드를 입력해주세요.";
    return;
  }

  // 이메일 형식 검증
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) {
    statusElement.innerText = "올바른 이메일 형식이 아닙니다.";
    return;
  }

  // 비밀번호 길이 검증
  if (password.length < 6) {
    statusElement.innerText = "비밀번호는 최소 6자 이상이어야 합니다.";
    return;
  }

  // 비밀번호 일치 검증
  if (password !== passwordConfirm) {
    statusElement.innerText = "비밀번호가 일치하지 않습니다.";
    return;
  }

  try {
    console.log("Firebase 회원가입 시도");
    // Firebase 회원가입
    const userCredential = await createUserWithEmailAndPassword(auth, email, password);
    console.log("회원가입 성공:", userCredential.user);

    // 회원가입 성공 시 자동 로그인되어 onAuthStateChanged에서 처리됨
    statusElement.innerText = "회원가입 성공!";
    statusElement.className = "success-text";

  } catch (error) {
    console.error("회원가입 실패:", error);

    // 에러 메시지 한글화
    let errorMessage = "회원가입 중 오류가 발생했습니다.";
    switch (error.code) {
      case 'auth/email-already-in-use':
        errorMessage = "이미 사용 중인 이메일입니다.";
        break;
      case 'auth/invalid-email':
        errorMessage = "올바르지 않은 이메일 형식입니다.";
        break;
      case 'auth/operation-not-allowed':
        errorMessage = "이메일/비밀번호 회원가입이 비활성화되어 있습니다.";
        break;
      case 'auth/weak-password':
        errorMessage = "비밀번호가 너무 약합니다.";
        break;
      default:
        errorMessage = `회원가입 실패: ${error.message}`;
    }

    statusElement.innerText = errorMessage;
    statusElement.className = "error-text";
  }
});

// 로그인 섹션에 비밀번호 초기화 링크 추가
document.getElementById("login-section").querySelector(".auth-switch").innerHTML += `
  <div style="margin-top: 8px;">
    비밀번호를 잊으셨나요? <a href="#" id="show-reset-password">비밀번호 초기화</a>
  </div>
`;

// 비밀번호 초기화 섹션 추가
const resetPasswordSection = document.createElement("div");
resetPasswordSection.id = "reset-password-section";
resetPasswordSection.className = "card";
resetPasswordSection.style.display = "none";
resetPasswordSection.innerHTML = `
  <h2>비밀번호 초기화</h2>
  <div class="input-group">
    <label for="reset-email">Email</label>
    <input type="email" id="reset-email" placeholder="이메일" />
  </div>
  <button id="reset-password-btn" class="btn-primary">초기화 메일 보내기</button>
  <div id="reset-status" class="error-text"></div>
  <div class="auth-switch">
    <a href="#" id="back-to-login">로그인으로 돌아가기</a>
  </div>
`;
document.querySelector(".container").appendChild(resetPasswordSection);

// 비밀번호 초기화 링크 클릭 이벤트
document.getElementById("show-reset-password").addEventListener("click", (e) => {
  e.preventDefault();
  document.getElementById("login-section").style.display = "none";
  document.getElementById("reset-password-section").style.display = "block";
});

// 로그인으로 돌아가기 링크 클릭 이벤트
document.getElementById("back-to-login").addEventListener("click", (e) => {
  e.preventDefault();
  document.getElementById("reset-password-section").style.display = "none";
  document.getElementById("login-section").style.display = "block";
});

// 비밀번호 초기화 버튼 클릭 이벤트
document.getElementById("reset-password-btn").addEventListener("click", async () => {
  const email = document.getElementById("reset-email").value;
  const statusElement = document.getElementById("reset-status");

  if (!email) {
    statusElement.innerText = "이메일을 입력해주세요.";
    return;
  }

  try {
    await sendPasswordResetEmail(auth, email);
    statusElement.innerText = "비밀번호 초기화 메일이 발송되었습니다. 이메일을 확인해주세요.";
    statusElement.className = "success-text";
  } catch (error) {
    console.error("비밀번호 초기화 실패:", error);

    let errorMessage = "비밀번호 초기화 중 오류가 발생했습니다.";
    switch (error.code) {
      case 'auth/invalid-email':
        errorMessage = "올바르지 않은 이메일 형식입니다.";
        break;
      case 'auth/user-not-found':
        errorMessage = "등록되지 않은 이메일입니다.";
        break;
    }

    statusElement.innerText = errorMessage;
    statusElement.className = "error-text";
  }
});

// 영상 추가 확인 섹션 추가
const addVideoSection = document.createElement("div");
addVideoSection.id = "add-video-section";
addVideoSection.className = "card";
addVideoSection.style.display = "none";
addVideoSection.innerHTML = `
  <h2>영상 추가</h2>
  <div class="video-preview">
    <img id="video-thumbnail" src="" alt="비디오 썸네일" style="width: 100%; border-radius: 4px; margin-bottom: 12px;">
    <div id="video-title" style="font-weight: bold; margin-bottom: 8px;"></div>
    <div id="video-artist" style="color: #666; margin-bottom: 4px;"></div>
    <div id="video-duration" style="color: #666; margin-bottom: 12px;"></div>
  </div>
  <div class="button-group" style="display: flex; gap: 8px;">
    <button id="confirm-add-btn" class="btn-primary">추가하기</button>
    <button id="cancel-add-btn" class="btn-secondary">취소</button>
  </div>
  <div id="add-status" class="error-text" style="margin-top: 8px;"></div>
`;
document.querySelector(".container").appendChild(addVideoSection);

// background script와 연결 관련 코드 수정
let port = null;

function connectToBackground() {
  try {
    port = chrome.runtime.connect({ name: 'popup' });

    port.onDisconnect.addListener(() => {
      console.log('[Connection] Background와 연결이 끊어짐');
      port = null;
    });

    // background로부터의 메시지 처리
    port.onMessage.addListener(async (message) => {
      if (message.action === 'ADD_TO_TEETUBE') {
        console.log('[ADD_TO_TEETUBE] 받은 비디오 ID:', message.videoId);
        await handleAddToTeetube(message.videoId);
      }
    });
  } catch (error) {
    console.error('[Connection] Background 연결 실패:', error);
  }
}

// 비디오 추가 처리를 별도 함수로 분리
async function handleAddToTeetube(videoId) {
  try {
    // 인증 상태가 준비될 때까지 잠시 대기
    await new Promise(resolve => setTimeout(resolve, 1000));

    // 현재 로그인 유저 확인
    const user = auth.currentUser;
    console.log('[ADD_TO_TEETUBE] 현재 유저 상태:', user ? '로그인됨' : '로그인되지 않음');

    if (!user) {
      // 로그인 섹션 표시
      document.getElementById('app-section').style.display = 'none';
      document.getElementById('add-video-section').style.display = 'none';
      document.getElementById('signup-section').style.display = 'none';
      document.getElementById('reset-password-section').style.display = 'none';
      document.getElementById('login-section').style.display = 'block';

      // 로그인 필요 메시지 표시
      const loginStatus = document.getElementById('login-status');
      loginStatus.innerText = '영상을 저장하려면 로그인이 필요합니다.';
      loginStatus.style.display = 'block';
      return;
    }

    // 이미 저장된 영상인지 체크
    const exists = await isVideoAlreadySaved(user.uid, videoId);
    if (exists) {
      showNotification('이미 저장된 동영상입니다');
      return;
    }

    // 메타데이터 가져오기
    const meta = await fetchYoutubeMetadata(videoId);
    currentVideoData = {
      ...meta,
      videoId: videoId,
      youtubeUrl: `https://www.youtube.com/watch?v=${videoId}`
    };

    // 영상 정보 표시
    document.getElementById('video-thumbnail').src = meta.thumbnailUrl;
    document.getElementById('video-title').textContent = meta.title;
    document.getElementById('video-artist').textContent = meta.artist;
    document.getElementById('video-duration').textContent = meta.duration;

    // 다른 섹션들 숨기기
    document.getElementById('login-section').style.display = 'none';
    document.getElementById('signup-section').style.display = 'none';
    document.getElementById('reset-password-section').style.display = 'none';
    document.getElementById('app-section').style.display = 'none';

    // 추가 확인 섹션 표시
    document.getElementById('add-video-section').style.display = 'block';
  } catch (error) {
    console.error('[ADD_TO_TEETUBE] 처리 중 오류 발생:', error);
    showNotification('영상 정보를 불러오는데 실패했습니다');
  }
}

// 페이지 로드 시 background와 연결 설정
document.addEventListener('DOMContentLoaded', () => {
  console.log('[Init] Popup 초기화 시작');
  connectToBackground();

  // pending video 확인
  if (chrome.runtime && chrome.runtime.sendMessage) {
    chrome.runtime.sendMessage({ action: 'GET_PENDING_VIDEO' }, async (response) => {
      if (chrome.runtime.lastError) {
        console.log('[Init] Pending video 확인 중 오류:', chrome.runtime.lastError);
        return;
      }

      if (response && response.videoId) {
        console.log('[Init] Pending video 발견:', response.videoId);
        await handleAddToTeetube(response.videoId);
      }
    });
  }
});

let currentVideoData = null;

// 추가 확인 버튼 클릭 이벤트 수정
document.getElementById('confirm-add-btn').addEventListener('click', async () => {
  if (!currentVideoData) return;

  const user = auth.currentUser;
  if (!user) {
    showNotification('로그인이 필요합니다');
    return;
  }

  try {
    // Firestore 문서 ID 생성
    const MAX_TIMESTAMP = 9999999999999;
    const docId = (MAX_TIMESTAMP - Date.now()).toString();

    // Firestore에 저장할 데이터
    const videoData = {
      artist: currentVideoData.artist,
      duration: currentVideoData.duration,
      id: currentVideoData.videoId,
      savedAt: new Date().toLocaleString("ko-KR", { timeZone: "Asia/Seoul" }),
      thumbnailUrl: currentVideoData.thumbnailUrl,
      timestamp: Date.now(),
      title: currentVideoData.title,
      userId: user.uid,
      youtubeUrl: currentVideoData.youtubeUrl,
      docId: docId
    };

    await saveVideoData(user.uid, docId, videoData);
    await loadDownloadQueue(user.uid);

    // 성공 메시지 표시
    showNotification('TeeTube에 추가되었습니다');

    // 앱 섹션으로 돌아가기
    document.getElementById('add-video-section').style.display = 'none';
    document.getElementById('app-section').style.display = 'flex';

    // 현재 비디오 데이터 초기화
    currentVideoData = null;

    // 성공 시 pending video 초기화
    chrome.runtime.sendMessage({ action: 'CLEAR_PENDING_VIDEO' });
  } catch (error) {
    console.error("[ADD_TO_TEETUBE] 저장 실패:", error);
    showNotification('저장에 실패했습니다');
  }
});

// 취소 버튼 클릭 이벤트 수정
document.getElementById('cancel-add-btn').addEventListener('click', () => {
  // pending video 초기화
  chrome.runtime.sendMessage({ action: 'CLEAR_PENDING_VIDEO' });

  // 앱 섹션으로 돌아가기
  document.getElementById('add-video-section').style.display = 'none';
  document.getElementById('app-section').style.display = 'flex';

  // 현재 비디오 데이터 초기화
  currentVideoData = null;
});

// 알림 메시지 표시 함수
function showNotification(message) {
  const status = document.getElementById('message');
  if (status) {
    status.innerText = message;
    status.style.display = 'block';
    setTimeout(() => {
      status.style.display = 'none';
    }, 3000);
  }
}
